# -*- coding: utf-8 -*-
"""
ELQE Analyser 2.1 Parameter Plotting Mod v1.2

For use with summary data in csv files generated by ELQE Analyser_v2_1.py

Developed by Emil Grove Dyrvik (Department of Physics, University of Oxford)
2020/2021

Contact:
emil.dyrvik@physics.ox.ac.uk
emil.dyrvik@gmail.com
"""
# TODO: For coming versions, consider creating, exporting, and closing one figure at the time before opening the next one. To reduce memory usage, avoid a warning due to more than 20 plots open, and possibly improve performance.

import numpy as np
import matplotlib.pyplot as plt
import matplotlib.cm as cm
import matplotlib as mpl
import seaborn as sns
import pandas as pd
import os
import datetime
import pathlib


""""Get the time"""
time_start = datetime.datetime.now()
date = datetime.date.today()

"""Global colourmaps for plots"""
CM_SUBSECTION_GREENS = np.linspace(0.4, 1.0, 1000) 
COLOURS_GREENS = [ cm.Greens(x) for x in CM_SUBSECTION_GREENS ]

CM_SUBSECTION_SPECTRAL = np.linspace(0, 1.0, 1000) 
COLOURS_SPECTRAL = [ cm.nipy_spectral(x) for x in CM_SUBSECTION_SPECTRAL ]

CM_SUBSECTION_ORANGES = np.linspace(0.3, 1.0, 1000) 
COLOURS_ORANGES = [ cm.Oranges(x) for x in CM_SUBSECTION_ORANGES ]

"""Global constant defaults (prompted for user definition at start)"""
MEASUREMENT_TYPE_DEFAULT = 1 # scans = 1, timed = 2

SCAN_SUMMARY_NAME_DEFAULT = 'igled19-batch5 scan measurements analysis summary 2021-03-22 param app.csv' # if 0 then ignore this data type
BIG_DATA_SUMMARY_NAME_DEFAULT = 0 # if 0 then ignore this data type
TIMED_SUMMARY_NAME_DEFAULT = 0 # if 0 then ignore this data type

PARAMETER_NAME_DEFAULT = 'ALD cycles' # name of the column with parameters to plot

EXPORT_FOLDER_NAME_DEFAULT = 'Parameter plots ' + date.strftime('%Y-%m-%d')

"""Prompt for measurement type, file origin, and analysis parameter values"""
def prompt_user_input(): # modified relative to the original
    """"Prompt if interested in analysing scans or timed measurements"""
    temporary_input = input('Select measurement type for analysis (scans = 1, timed = 2): ')
    if int(temporary_input) == 99: # shortcut to default everything
        return [MEASUREMENT_TYPE_DEFAULT,\
                SCAN_SUMMARY_NAME_DEFAULT,\
                BIG_DATA_SUMMARY_NAME_DEFAULT,\
                TIMED_SUMMARY_NAME_DEFAULT,\
                PARAMETER_NAME_DEFAULT,\
                EXPORT_FOLDER_NAME_DEFAULT]
    elif int(temporary_input) == 1 or int(temporary_input) == 2:
        measurement_type = int(temporary_input)
    else:
        measurement_type = MEASUREMENT_TYPE_DEFAULT
    
    """"Setting function-internal booleans for the different measurement types (scan/timed)"""
    if measurement_type == 1:
        type_scan_internal = True
    elif measurement_type == 2:
        type_scan_internal = False
    type_timed_internal = not type_scan_internal
    
    """Ask where the data is stored"""
    if type_scan_internal:
        temporary_string = 'Enter name of scan summary csv file (must be in the same location as the script), or enter 0 to ignore this data type: '
        temporary_input = input(temporary_string)
        if temporary_input == '0':
            scan_summary_name = 0
        else:
           scan_summary_name = temporary_input
            
        temporary_string = 'Enter name of scan big data summary csv file (must be in the same location as the script), or enter 0 to ignore this data type: '
        temporary_input = input(temporary_string)
        if temporary_input == '0':
            big_data_summary_name = 0
        else:
           big_data_summary_name = temporary_input
        
        timed_summary_name = 0
    if type_timed_internal:
        temporary_string = 'Enter name of timed summary csv file (must be in the same location as the script): '
        temporary_input = input(temporary_string)
        timed_summary_name = temporary_input
        scan_summary_name = 0
        big_data_summary_name = 0
        
    """Ask for the name of the column of the parameter of interest in the csv file"""
    temporary_string = 'Enter name of the column of the parameter of interest: '
    temporary_input = input(temporary_string)
    parameter_name = temporary_input
    
    """Possibility to set export folder has been removed for now because of Adobe Acrobat having an issue with file paths being too long. Therefore removing one folder from the hierarchy to generally shorten file lengths. Plots will still export to subfolders"""
# =============================================================================
#     """Ask where the plots should be exported"""   
#     temporary_string = 'Determine a name for the export folder, or enter 0 for default (' + EXPORT_FOLDER_NAME_DEFAULT + '): '
#     temporary_input = input(temporary_string)
#     if len(temporary_input) == 1:
#         if int(temporary_input) == 0:
#             export_folder_name = EXPORT_FOLDER_NAME_DEFAULT
#     else:
#         export_folder_name = temporary_input
# =============================================================================
    export_folder_name = EXPORT_FOLDER_NAME_DEFAULT
    

    return [measurement_type,\
            scan_summary_name,\
            big_data_summary_name,\
            timed_summary_name,\
            parameter_name,\
            export_folder_name]
    
"""Write one or multiple strings to the log file"""
def write_to_log(to_log):
    """to_log is either a string or a list of strings, without newline character \n"""
    log_file = open(log_file_path,'a')
    if isinstance(to_log,str):
        string = to_log + '\n'
        log_file.write(string)
    else:
        for i in range(len(to_log)):
            string = to_log[i] + '\n'
            log_file.write(string)
            
    log_file.close()

"""Plotting summary plots for the scan analysis"""
def plot_scan_summary(scan_summary_df,experiment_name): # modified relative to the original
    """Some constants for graphical settings"""
    scalar_ticksize_major = 5
    scalar_ticksize_minor = 0
    scalar_tickwidth_minor = 0
    scalar_linewidth = 2
    
    """Box swarm plot general graphic settings"""
    boxcolour = COLOURS_GREENS[700]
    boxprops = {'edgecolor': boxcolour, 'linewidth': scalar_linewidth, 'facecolor': 'w'}
    lineprops = {'color': boxcolour, 'linewidth': scalar_linewidth}
    boxplot_kwargs = dict({'boxprops': boxprops, 'medianprops': lineprops,
                           'whiskerprops': lineprops, 'capprops': lineprops,
                           'width': 0.75})
    
    """Plotting all the box swarm plots"""
    """Peak EQE"""
    fig_peak_eqe = plt.figure(1)
    fig_peak_eqe.set_size_inches(9,8)
    fig_peak_eqe.set_tight_layout(True)
    ax_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Peak EQE [%]'].notna()], y=scan_summary_df['Peak EQE [%]'][scan_summary_df['Peak EQE [%]'].notna()], ax=ax_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Peak EQE [%]'].notna()], y=scan_summary_df['Peak EQE [%]'][scan_summary_df['Peak EQE [%]'].notna()], ax=ax_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_peak_eqe.set_xlabel(parameter_name)
    ax_peak_eqe.set_ylabel('Peak EQE [%]')
    ax_peak_eqe.set_title(experiment_name)
    ax_peak_eqe.set_ylim(bottom=0)
    if scan_summary_df['Peak EQE [%]'].max() > 25:
        ax_peak_eqe.set_ylim(top=25)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
        
    """Voltage @ Peak EQE"""
    fig_voltage_at_peak_eqe = plt.figure(2)
    fig_voltage_at_peak_eqe.set_size_inches(9,8)
    fig_voltage_at_peak_eqe.set_tight_layout(True)
    ax_voltage_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Peak EQE [V]'].notna()], y=scan_summary_df['Voltage at Peak EQE [V]'][scan_summary_df['Voltage at Peak EQE [V]'].notna()], ax=ax_voltage_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Peak EQE [V]'].notna()], y=scan_summary_df['Voltage at Peak EQE [V]'][scan_summary_df['Voltage at Peak EQE [V]'].notna()], ax=ax_voltage_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_peak_eqe.set_ylim(bottom=0)
    ax_voltage_at_peak_eqe.set_xlabel(parameter_name)
    ax_voltage_at_peak_eqe.set_ylabel('Voltage at Peak EQE [V]')
    ax_voltage_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Peak EQE"""
    fig_current_at_peak_eqe = plt.figure(3)
    fig_current_at_peak_eqe.set_size_inches(9,8)
    fig_current_at_peak_eqe.set_tight_layout(True)
    ax_current_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak EQE [mA/cm2]'][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], ax=ax_current_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak EQE [mA/cm2]'][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], ax=ax_current_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_peak_eqe.set_ylim(bottom=0)
    ax_current_at_peak_eqe.set_xlabel(parameter_name)
    ax_current_at_peak_eqe.set_ylabel('Current Density at Peak EQE [mA/cm2]')
    ax_current_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Radiance @ Peak EQE"""
    fig_radiance_at_peak_eqe = plt.figure(4)
    fig_radiance_at_peak_eqe.set_size_inches(9,8)
    fig_radiance_at_peak_eqe.set_tight_layout(True)
    ax_radiance_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], y=scan_summary_df['Radiance at Peak EQE [W/sr/m2]'][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], ax=ax_radiance_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], y=scan_summary_df['Radiance at Peak EQE [W/sr/m2]'][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], ax=ax_radiance_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_radiance_at_peak_eqe.set_ylim(bottom=0)
    ax_radiance_at_peak_eqe.set_xlabel(parameter_name)
    ax_radiance_at_peak_eqe.set_ylabel('Radiance at Peak EQE [W/sr/m2]')
    ax_radiance_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Scan # @ Peak EQE"""
    fig_scan_at_peak_eqe = plt.figure(5)
    fig_scan_at_peak_eqe.set_size_inches(9,8)
    fig_scan_at_peak_eqe.set_tight_layout(True)
    ax_scan_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Scan # at Peak EQE'].notna()], y=scan_summary_df['Scan # at Peak EQE'][scan_summary_df['Scan # at Peak EQE'].notna()], ax=ax_scan_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Scan # at Peak EQE'].notna()], y=scan_summary_df['Scan # at Peak EQE'][scan_summary_df['Scan # at Peak EQE'].notna()], ax=ax_scan_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_scan_at_peak_eqe.set_ylim(bottom=0)
    ax_scan_at_peak_eqe.set_xlabel(parameter_name)
    ax_scan_at_peak_eqe.set_ylabel('Scan # at Peak EQE')
    ax_scan_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Radiance @ Luminescence Turn-on (Peak EQE scan)"""
    fig_radiance_at_turn_on_peak_eqe_scan = plt.figure(6)
    fig_radiance_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_radiance_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_radiance_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], ax=ax_radiance_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], ax=ax_radiance_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_radiance_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_radiance_at_turn_on_peak_eqe_scan.set_xlabel(parameter_name)
    ax_radiance_at_turn_on_peak_eqe_scan.set_ylabel('Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)')
    ax_radiance_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_radiance_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_radiance_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Voltage @ Luminescence Turn-on (Peak EQE scan)"""
    fig_voltage_at_turn_on_peak_eqe_scan = plt.figure(7)
    fig_voltage_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_voltage_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_voltage_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], ax=ax_voltage_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], ax=ax_voltage_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_voltage_at_turn_on_peak_eqe_scan.set_xlabel(parameter_name)
    ax_voltage_at_turn_on_peak_eqe_scan.set_ylabel('Voltage at Luminescence Turn-on [V] (Peak EQE scan)')
    ax_voltage_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_voltage_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_voltage_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Luminescence Turn-on (Peak EQE scan)"""
    fig_current_at_turn_on_peak_eqe_scan = plt.figure(8)
    fig_current_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_current_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_current_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], ax=ax_current_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], ax=ax_current_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_current_at_turn_on_peak_eqe_scan.set_xlabel(parameter_name)
    ax_current_at_turn_on_peak_eqe_scan.set_ylabel('Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)')
    ax_current_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_current_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_current_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """EQE @ Luminescence Turn-on (Peak EQE scan)"""
    fig_eqe_at_turn_on_peak_eqe_scan = plt.figure(9)
    fig_eqe_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_eqe_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_eqe_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], ax=ax_eqe_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], ax=ax_eqe_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_eqe_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_eqe_at_turn_on_peak_eqe_scan.set_xlabel(parameter_name)
    ax_eqe_at_turn_on_peak_eqe_scan.set_ylabel('EQE at Luminescence Turn-on [%] (Peak EQE scan)')
    ax_eqe_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_eqe_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_eqe_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    if scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].max() > 25:
        ax_eqe_at_turn_on_peak_eqe_scan.set_ylim(top=25)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Peak Radiance"""
    fig_peak_radiance = plt.figure(10)
    fig_peak_radiance.set_size_inches(9,8)
    fig_peak_radiance.set_tight_layout(True)
    ax_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], y=scan_summary_df['Peak Radiance [W/sr/m2]'][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], ax=ax_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], y=scan_summary_df['Peak Radiance [W/sr/m2]'][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], ax=ax_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_peak_radiance.set_ylim(bottom=0)
    ax_peak_radiance.set_xlabel(parameter_name)
    ax_peak_radiance.set_ylabel('Peak Radiance [W/sr/m2]')
    ax_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Voltage @ Peak Radiance"""
    fig_voltage_at_peak_radiance = plt.figure(11)
    fig_voltage_at_peak_radiance.set_size_inches(9,8)
    fig_voltage_at_peak_radiance.set_tight_layout(True)
    ax_voltage_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], y=scan_summary_df['Voltage at Peak Radiance [V]'][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], ax=ax_voltage_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], y=scan_summary_df['Voltage at Peak Radiance [V]'][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], ax=ax_voltage_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_peak_radiance.set_ylim(bottom=0)
    ax_voltage_at_peak_radiance.set_xlabel(parameter_name)
    ax_voltage_at_peak_radiance.set_ylabel('Voltage at Peak Radiance [V]')
    ax_voltage_at_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Peak Radiance"""
    fig_current_at_peak_radiance = plt.figure(12)
    fig_current_at_peak_radiance.set_size_inches(9,8)
    fig_current_at_peak_radiance.set_tight_layout(True)
    ax_current_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak Radiance [mA/cm2]'][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], ax=ax_current_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak Radiance [mA/cm2]'][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], ax=ax_current_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_peak_radiance.set_ylim(bottom=0)
    ax_current_at_peak_radiance.set_xlabel(parameter_name)
    ax_current_at_peak_radiance.set_ylabel('Current Density at Peak Radiance [mA/cm2]')
    ax_current_at_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """EQE @ Peak Radiance"""
    fig_eqe_at_peak_radiance = plt.figure(13)
    fig_eqe_at_peak_radiance.set_size_inches(9,8)
    fig_eqe_at_peak_radiance.set_tight_layout(True)
    ax_eqe_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['EQE at Peak Radiance [%]'].notna()], y=scan_summary_df['EQE at Peak Radiance [%]'][scan_summary_df['EQE at Peak Radiance [%]'].notna()], ax=ax_eqe_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['EQE at Peak Radiance [%]'].notna()], y=scan_summary_df['EQE at Peak Radiance [%]'][scan_summary_df['EQE at Peak Radiance [%]'].notna()], ax=ax_eqe_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_eqe_at_peak_radiance.set_ylim(bottom=0)
    ax_eqe_at_peak_radiance.set_xlabel(parameter_name)
    ax_eqe_at_peak_radiance.set_ylabel('EQE at Peak Radiance [%]')
    ax_eqe_at_peak_radiance.set_title(experiment_name)
    if scan_summary_df['EQE at Peak Radiance [%]'].max() > 25:
        ax_eqe_at_peak_radiance.set_ylim(top=25)    
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Scan # @ Peak Radiance"""
    fig_scan_at_peak_radiance = plt.figure(14)
    fig_scan_at_peak_radiance.set_size_inches(9,8)
    fig_scan_at_peak_radiance.set_tight_layout(True)
    ax_scan_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Scan # at Peak Radiance'].notna()], y=scan_summary_df['Scan # at Peak Radiance'][scan_summary_df['Scan # at Peak Radiance'].notna()], ax=ax_scan_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Scan # at Peak Radiance'].notna()], y=scan_summary_df['Scan # at Peak Radiance'][scan_summary_df['Scan # at Peak Radiance'].notna()], ax=ax_scan_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_scan_at_peak_radiance.set_ylim(bottom=0)
    ax_scan_at_peak_radiance.set_xlabel(parameter_name)
    ax_scan_at_peak_radiance.set_ylabel('Scan # at Peak Radiance')
    ax_scan_at_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Radiance @ Luminescence Turn-on (Peak radiance scan)"""
    fig_radiance_at_turn_on_peak_radiance_scan = plt.figure(15)
    fig_radiance_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_radiance_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_radiance_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], ax=ax_radiance_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], ax=ax_radiance_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_radiance_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_radiance_at_turn_on_peak_radiance_scan.set_xlabel(parameter_name)
    ax_radiance_at_turn_on_peak_radiance_scan.set_ylabel('Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)')
    ax_radiance_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_radiance_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_radiance_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Voltage @ Luminescence Turn-on (Peak radiance scan)"""
    fig_voltage_at_turn_on_peak_radiance_scan = plt.figure(16)
    fig_voltage_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_voltage_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_voltage_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], ax=ax_voltage_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], ax=ax_voltage_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_voltage_at_turn_on_peak_radiance_scan.set_xlabel(parameter_name)
    ax_voltage_at_turn_on_peak_radiance_scan.set_ylabel('Voltage at Luminescence Turn-on [V] (Peak radiance scan)')
    ax_voltage_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_voltage_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_voltage_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Luminescence Turn-on (Peak radiance scan)"""
    fig_current_at_turn_on_peak_radiance_scan = plt.figure(17)
    fig_current_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_current_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_current_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], ax=ax_current_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], ax=ax_current_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_current_at_turn_on_peak_radiance_scan.set_xlabel(parameter_name)
    ax_current_at_turn_on_peak_radiance_scan.set_ylabel('Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)')
    ax_current_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_current_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_current_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """EQE @ Luminescence Turn-on (Peak radiance scan)"""
    fig_eqe_at_turn_on_peak_radiance_scan = plt.figure(18)
    fig_eqe_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_eqe_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_eqe_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df[parameter_name][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], ax=ax_eqe_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df[parameter_name][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], ax=ax_eqe_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_eqe_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_eqe_at_turn_on_peak_radiance_scan.set_xlabel(parameter_name)
    ax_eqe_at_turn_on_peak_radiance_scan.set_ylabel('EQE at Luminescence Turn-on [%] (Peak radiance scan)')
    ax_eqe_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_eqe_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_eqe_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    if scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].max() > 25:
        ax_eqe_at_turn_on_peak_radiance_scan.set_ylim(top=25)    
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth
    
    ax_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_voltage_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_current_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_radiance_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_scan_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_scan_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_radiance_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_voltage_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_current_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_eqe_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_voltage_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_current_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_eqe_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_scan_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_scan_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_radiance_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_voltage_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_current_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_eqe_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)           
    
    """Exporting plots"""
    plot_name = 'Peak EQE box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_peak_eqe.savefig(pdf_path)
    fig_peak_eqe.savefig(svg_path)
    plt.close(fig_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = 'V at Peak EQE box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_peak_eqe.savefig(pdf_path)
    fig_voltage_at_peak_eqe.savefig(svg_path)
    plt.close(fig_voltage_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = 'J at Peak EQE box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_peak_eqe.savefig(pdf_path)
    fig_current_at_peak_eqe.savefig(svg_path)
    plt.close(fig_current_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'Rad at Peak EQE box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_radiance_at_peak_eqe.savefig(pdf_path)
    fig_radiance_at_peak_eqe.savefig(svg_path)
    plt.close(fig_radiance_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'Scan # at Peak EQE box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_scan_at_peak_eqe.savefig(pdf_path)
    fig_scan_at_peak_eqe.savefig(svg_path)
    plt.close(fig_scan_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'Rad at Lum Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_radiance_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_radiance_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_radiance_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'V at Lum Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_voltage_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_voltage_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'J at Lum Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_current_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_current_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'EQE at Lum Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_eqe_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_eqe_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_eqe_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'Peak Rad box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_peak_radiance.savefig(pdf_path)
    fig_peak_radiance.savefig(svg_path)
    plt.close(fig_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'V at Peak Rad box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_peak_radiance.savefig(pdf_path)
    fig_voltage_at_peak_radiance.savefig(svg_path)
    plt.close(fig_voltage_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'J at Peak Rad box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_peak_radiance.savefig(pdf_path)
    fig_current_at_peak_radiance.savefig(svg_path)
    plt.close(fig_current_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'EQE at Peak Rad box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_eqe_at_peak_radiance.savefig(pdf_path)
    fig_eqe_at_peak_radiance.savefig(svg_path)
    plt.close(fig_eqe_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'Scan # at Peak Rad box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_scan_at_peak_radiance.savefig(pdf_path)
    fig_scan_at_peak_radiance.savefig(svg_path)
    plt.close(fig_scan_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'Rad at Lum Turn-on (Peak Rad scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_radiance_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_radiance_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_radiance_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'V at Lum Turn-on (Peak Rad scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_voltage_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_voltage_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'J at Lum Turn-on (Peak Rad scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_current_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_current_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = 'EQE at Lum Turn-on (Peak Rad scan) box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_eqe_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_eqe_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_eqe_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    print('Exported summary .pdf and .svg plots for: %s' % experiment_name)

"""Plotting big data plots for scan analysis"""
def plot_big_data(big_data_df,experiment_name): # modified relative to the original
    # The box-stripplots are sometimes very narrow, which I think is related to this: https://stackoverflow.com/a/56357825
    """Some constants for graphical settings"""
    scalar_ticksize_major = 5
    scalar_ticksize_minor = 0
    scalar_tickwidth_minor = 0
    scalar_linewidth = 2
    
    """Box strip plot general graphic settings"""
    boxcolour = COLOURS_GREENS[700]
    boxprops = {'edgecolor': boxcolour, 'linewidth': scalar_linewidth, 'facecolor': 'w'}
    lineprops = {'color': boxcolour, 'linewidth': scalar_linewidth}
    boxplot_kwargs = dict({'boxprops': boxprops, 'medianprops': lineprops,
                           'whiskerprops': lineprops, 'capprops': lineprops,
                           'width': 0.75})
    
    """Plotting all the box-stripplots"""
    """Limiting the number of open plots at the time to approximately 10 or less"""
    
    """Voltage at Max EQE"""
    fig_voltage_at_max_eqe = plt.figure(1)
    fig_voltage_at_max_eqe.set_size_inches(12,8)
    fig_voltage_at_max_eqe.set_tight_layout(True)
    ax_voltage_at_max_eqe = plt.gca()  
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at Max EQE [V]'], ax=ax_voltage_at_max_eqe, color='k', size=4)
    ax_voltage_at_max_eqe.set_xlabel(parameter_name)
    ax_voltage_at_max_eqe.set_ylabel('Voltage at Max EQE [V]')
    ax_voltage_at_max_eqe.set_title(experiment_name)
    ax_voltage_at_max_eqe.set_ylim(bottom=0)
    
    """Current Density at Max EQE"""
    fig_current_at_max_eqe = plt.figure(2)
    fig_current_at_max_eqe.set_size_inches(12,8)
    fig_current_at_max_eqe.set_tight_layout(True)
    ax_current_at_max_eqe = plt.gca()  
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at Max EQE [mA/cm2]'], ax=ax_current_at_max_eqe, color='k', size=4)
    ax_current_at_max_eqe.set_xlabel(parameter_name)
    ax_current_at_max_eqe.set_ylabel('Current Density at Max EQE [mA/cm2]')
    ax_current_at_max_eqe.set_title(experiment_name)
    ax_current_at_max_eqe.set_ylim(bottom=0)    
    
    """Radiance at Max EQE"""
    fig_radiance_at_max_eqe = plt.figure(3)
    fig_radiance_at_max_eqe.set_size_inches(12,8)
    fig_radiance_at_max_eqe.set_tight_layout(True)
    ax_radiance_at_max_eqe = plt.gca()  
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at Max EQE [W/sr/m2]'], ax=ax_radiance_at_max_eqe, color='k', size=4)
    ax_radiance_at_max_eqe.set_xlabel(parameter_name)
    ax_radiance_at_max_eqe.set_ylabel('Radiance at Max EQE [W/sr/m2]')
    ax_radiance_at_max_eqe.set_title(experiment_name)
    ax_radiance_at_max_eqe.set_ylim(bottom=0)
    
    """EQE at Max EQE"""
    fig_eqe_at_max_eqe = plt.figure(4)
    fig_eqe_at_max_eqe.set_size_inches(12,8)
    fig_eqe_at_max_eqe.set_tight_layout(True)
    ax_eqe_at_max_eqe = plt.gca()  
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at Max EQE [%]'], ax=ax_eqe_at_max_eqe, color='k', size=4)
    ax_eqe_at_max_eqe.set_xlabel(parameter_name)
    ax_eqe_at_max_eqe.set_ylabel('EQE at Max EQE [%]')
    ax_eqe_at_max_eqe.set_title(experiment_name)
    ax_eqe_at_max_eqe.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth
    
    ax_voltage_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_current_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_radiance_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_eqe_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    """Exporting plots"""    
    plot_name = 'V at Max EQE box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_max_eqe.savefig(pdf_path)
    fig_voltage_at_max_eqe.savefig(svg_path)
    plt.close(fig_voltage_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'J at Max EQE box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_max_eqe.savefig(pdf_path)
    fig_current_at_max_eqe.savefig(svg_path)
    plt.close(fig_current_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Rad at Max EQE box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_max_eqe.savefig(pdf_path)
    fig_radiance_at_max_eqe.savefig(svg_path)
    plt.close(fig_radiance_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))
    
    plot_name = 'EQE at Max EQE box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_max_eqe.savefig(pdf_path)
    fig_eqe_at_max_eqe.savefig(svg_path)
    plt.close(fig_eqe_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    """Voltage at 2.0 V"""
    fig_voltage_at_2p0_V = plt.figure(5)
    fig_voltage_at_2p0_V.set_size_inches(12,8)
    fig_voltage_at_2p0_V.set_tight_layout(True)
    ax_voltage_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 2.0 V [V]'], ax=ax_voltage_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 2.0 V [V]'], ax=ax_voltage_at_2p0_V, color='k', size=4)
    ax_voltage_at_2p0_V.set_xlabel(parameter_name)
    ax_voltage_at_2p0_V.set_ylabel('Voltage at 2.0 V [V]')
    ax_voltage_at_2p0_V.set_title(experiment_name)
    
    """Current Density at 2.0 V"""
    fig_current_at_2p0_V = plt.figure(6)
    fig_current_at_2p0_V.set_size_inches(12,8)
    fig_current_at_2p0_V.set_tight_layout(True)
    ax_current_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 2.0 V [mA/cm2]'], ax=ax_current_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 2.0 V [mA/cm2]'], ax=ax_current_at_2p0_V, color='k', size=4)
    ax_current_at_2p0_V.set_xlabel(parameter_name)
    ax_current_at_2p0_V.set_ylabel('Current Density at 2.0 V [mA/cm2]')
    ax_current_at_2p0_V.set_title(experiment_name)
    ax_current_at_2p0_V.set_ylim(bottom=0)
    
    """Radiance at 2.0 V"""
    fig_radiance_at_2p0_V = plt.figure(7)
    fig_radiance_at_2p0_V.set_size_inches(12,8)
    fig_radiance_at_2p0_V.set_tight_layout(True)
    ax_radiance_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 2.0 V [W/sr/m2]'], ax=ax_radiance_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 2.0 V [W/sr/m2]'], ax=ax_radiance_at_2p0_V, color='k', size=4)
    ax_radiance_at_2p0_V.set_xlabel(parameter_name)
    ax_radiance_at_2p0_V.set_ylabel('Radiance at 2.0 V [W/sr/m2]')
    ax_radiance_at_2p0_V.set_title(experiment_name)
    ax_radiance_at_2p0_V.set_ylim(bottom=0)
    
    """Luminance at 2.0 V"""
    fig_luminance_at_2p0_V = plt.figure(8)
    fig_luminance_at_2p0_V.set_size_inches(12,8)
    fig_luminance_at_2p0_V.set_tight_layout(True)
    ax_luminance_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 2.0 V [cd/m2]'], ax=ax_luminance_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 2.0 V [cd/m2]'], ax=ax_luminance_at_2p0_V, color='k', size=4)
    ax_luminance_at_2p0_V.set_xlabel(parameter_name)
    ax_luminance_at_2p0_V.set_ylabel('Luminance at 2.0 V [cd/m2]')
    ax_luminance_at_2p0_V.set_title(experiment_name)
    ax_luminance_at_2p0_V.set_ylim(bottom=0)
    
    """EQE at 2.0 V"""
    fig_eqe_at_2p0_V = plt.figure(9)
    fig_eqe_at_2p0_V.set_size_inches(12,8)
    fig_eqe_at_2p0_V.set_tight_layout(True)
    ax_eqe_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 2.0 V [%]'], ax=ax_eqe_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 2.0 V [%]'], ax=ax_eqe_at_2p0_V, color='k', size=4)
    ax_eqe_at_2p0_V.set_xlabel(parameter_name)
    ax_eqe_at_2p0_V.set_ylabel('EQE at 2.0 V [%]')
    ax_eqe_at_2p0_V.set_title(experiment_name)
    ax_eqe_at_2p0_V.set_ylim(bottom=0)
    
    """Current efficiency at 2.0 V"""
    fig_curr_eff_at_2p0_V = plt.figure(10)
    fig_curr_eff_at_2p0_V.set_size_inches(12,8)
    fig_curr_eff_at_2p0_V.set_tight_layout(True)
    ax_curr_eff_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 2.0 V [cd/A]'], ax=ax_curr_eff_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 2.0 V [cd/A]'], ax=ax_curr_eff_at_2p0_V, color='k', size=4)
    ax_curr_eff_at_2p0_V.set_xlabel(parameter_name)
    ax_curr_eff_at_2p0_V.set_ylabel('Current efficiency at 2.0 V [cd/A]')
    ax_curr_eff_at_2p0_V.set_title(experiment_name)
    ax_curr_eff_at_2p0_V.set_ylim(bottom=0)
    
    """Wall-plug efficiency at 2.0 V"""
    fig_WP_eff_at_2p0_V = plt.figure(11)
    fig_WP_eff_at_2p0_V.set_size_inches(12,8)
    fig_WP_eff_at_2p0_V.set_tight_layout(True)
    ax_WP_eff_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 2.0 V [W/W]'], ax=ax_WP_eff_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 2.0 V [W/W]'], ax=ax_WP_eff_at_2p0_V, color='k', size=4)
    ax_WP_eff_at_2p0_V.set_xlabel(parameter_name)
    ax_WP_eff_at_2p0_V.set_ylabel('Wall-plug efficiency at 2.0 V [W/W]')
    ax_WP_eff_at_2p0_V.set_title(experiment_name)
    ax_WP_eff_at_2p0_V.set_ylim(bottom=0)
    
    """Luminous efficacy at 2.0 V"""
    fig_lum_eff_at_2p0_V = plt.figure(12)
    fig_lum_eff_at_2p0_V.set_size_inches(12,8)
    fig_lum_eff_at_2p0_V.set_tight_layout(True)
    ax_lum_eff_at_2p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 2.0 V [lm/W]'], ax=ax_lum_eff_at_2p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 2.0 V [lm/W]'], ax=ax_lum_eff_at_2p0_V, color='k', size=4)
    ax_lum_eff_at_2p0_V.set_xlabel(parameter_name)
    ax_lum_eff_at_2p0_V.set_ylabel('Luminous efficacy at 2.0 V [lm/W]')
    ax_lum_eff_at_2p0_V.set_title(experiment_name)
    ax_lum_eff_at_2p0_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth 
    
    ax_voltage_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_current_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_radiance_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_luminance_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_luminance_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_eqe_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_curr_eff_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_curr_eff_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_WP_eff_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_WP_eff_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_lum_eff_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_lum_eff_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    """Exporting plots"""
    plot_name = 'V at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_2p0_V.savefig(pdf_path)
    fig_voltage_at_2p0_V.savefig(svg_path)
    plt.close(fig_voltage_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'J at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_2p0_V.savefig(pdf_path)
    fig_current_at_2p0_V.savefig(svg_path)
    plt.close(fig_current_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'Rad at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_2p0_V.savefig(pdf_path)
    fig_radiance_at_2p0_V.savefig(svg_path)
    plt.close(fig_radiance_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_luminance_at_2p0_V.savefig(pdf_path)
    fig_luminance_at_2p0_V.savefig(svg_path)
    plt.close(fig_luminance_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'EQE at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_2p0_V.savefig(pdf_path)
    fig_eqe_at_2p0_V.savefig(svg_path)
    plt.close(fig_eqe_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path)) 
    
    plot_name = 'Curr eff at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_curr_eff_at_2p0_V.savefig(pdf_path)
    fig_curr_eff_at_2p0_V.savefig(svg_path)
    plt.close(fig_curr_eff_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'WP eff at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_WP_eff_at_2p0_V.savefig(pdf_path)
    fig_WP_eff_at_2p0_V.savefig(svg_path)
    plt.close(fig_WP_eff_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum eff at 2p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_lum_eff_at_2p0_V.savefig(pdf_path)
    fig_lum_eff_at_2p0_V.savefig(svg_path)
    plt.close(fig_lum_eff_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    """Voltage at 2.5 V"""
    fig_voltage_at_2p5_V = plt.figure(13)
    fig_voltage_at_2p5_V.set_size_inches(12,8)
    fig_voltage_at_2p5_V.set_tight_layout(True)
    ax_voltage_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 2.5 V [V]'], ax=ax_voltage_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 2.5 V [V]'], ax=ax_voltage_at_2p5_V, color='k', size=4)
    ax_voltage_at_2p5_V.set_xlabel(parameter_name)
    ax_voltage_at_2p5_V.set_ylabel('Voltage at 2.5 V [V]')
    ax_voltage_at_2p5_V.set_title(experiment_name)
    
    """Current Density at 2.5 V"""
    fig_current_at_2p5_V = plt.figure(14)
    fig_current_at_2p5_V.set_size_inches(12,8)
    fig_current_at_2p5_V.set_tight_layout(True)
    ax_current_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 2.5 V [mA/cm2]'], ax=ax_current_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 2.5 V [mA/cm2]'], ax=ax_current_at_2p5_V, color='k', size=4)
    ax_current_at_2p5_V.set_xlabel(parameter_name)
    ax_current_at_2p5_V.set_ylabel('Current Density at 2.5 V [mA/cm2]')
    ax_current_at_2p5_V.set_title(experiment_name)
    ax_current_at_2p5_V.set_ylim(bottom=0)
    
    """Radiance at 2.5 V"""
    fig_radiance_at_2p5_V = plt.figure(15)
    fig_radiance_at_2p5_V.set_size_inches(12,8)
    fig_radiance_at_2p5_V.set_tight_layout(True)
    ax_radiance_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 2.5 V [W/sr/m2]'], ax=ax_radiance_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 2.5 V [W/sr/m2]'], ax=ax_radiance_at_2p5_V, color='k', size=4)
    ax_radiance_at_2p5_V.set_xlabel(parameter_name)
    ax_radiance_at_2p5_V.set_ylabel('Radiance at 2.5 V [W/sr/m2]')
    ax_radiance_at_2p5_V.set_title(experiment_name)
    ax_radiance_at_2p5_V.set_ylim(bottom=0)
    
    """Luminance at 2.5 V"""
    fig_luminance_at_2p5_V = plt.figure(16)
    fig_luminance_at_2p5_V.set_size_inches(12,8)
    fig_luminance_at_2p5_V.set_tight_layout(True)
    ax_luminance_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 2.5 V [cd/m2]'], ax=ax_luminance_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 2.5 V [cd/m2]'], ax=ax_luminance_at_2p5_V, color='k', size=4)
    ax_luminance_at_2p5_V.set_xlabel(parameter_name)
    ax_luminance_at_2p5_V.set_ylabel('Luminance at 2.5 V [cd/m2]')
    ax_luminance_at_2p5_V.set_title(experiment_name)
    ax_luminance_at_2p5_V.set_ylim(bottom=0)
    
    """EQE at 2.5 V"""
    fig_eqe_at_2p5_V = plt.figure(17)
    fig_eqe_at_2p5_V.set_size_inches(12,8)
    fig_eqe_at_2p5_V.set_tight_layout(True)
    ax_eqe_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 2.5 V [%]'], ax=ax_eqe_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 2.5 V [%]'], ax=ax_eqe_at_2p5_V, color='k', size=4)
    ax_eqe_at_2p5_V.set_xlabel(parameter_name)
    ax_eqe_at_2p5_V.set_ylabel('EQE at 2.5 V [%]')
    ax_eqe_at_2p5_V.set_title(experiment_name)
    ax_eqe_at_2p5_V.set_ylim(bottom=0)
    
    """Current efficiency at 2.5 V"""
    fig_curr_eff_at_2p5_V = plt.figure(18)
    fig_curr_eff_at_2p5_V.set_size_inches(12,8)
    fig_curr_eff_at_2p5_V.set_tight_layout(True)
    ax_curr_eff_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 2.5 V [cd/A]'], ax=ax_curr_eff_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 2.5 V [cd/A]'], ax=ax_curr_eff_at_2p5_V, color='k', size=4)
    ax_curr_eff_at_2p5_V.set_xlabel(parameter_name)
    ax_curr_eff_at_2p5_V.set_ylabel('Current efficiency at 2.5 V [cd/A]')
    ax_curr_eff_at_2p5_V.set_title(experiment_name)
    ax_curr_eff_at_2p5_V.set_ylim(bottom=0)
    
    """Wall-plug efficiency at 2.5 V"""
    fig_WP_eff_at_2p5_V = plt.figure(19)
    fig_WP_eff_at_2p5_V.set_size_inches(12,8)
    fig_WP_eff_at_2p5_V.set_tight_layout(True)
    ax_WP_eff_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 2.5 V [W/W]'], ax=ax_WP_eff_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 2.5 V [W/W]'], ax=ax_WP_eff_at_2p5_V, color='k', size=4)
    ax_WP_eff_at_2p5_V.set_xlabel(parameter_name)
    ax_WP_eff_at_2p5_V.set_ylabel('Wall-plug efficiency at 2.5 V [W/W]')
    ax_WP_eff_at_2p5_V.set_title(experiment_name)
    ax_WP_eff_at_2p5_V.set_ylim(bottom=0)
    
    """Luminous efficacy at 2.5 V"""
    fig_lum_eff_at_2p5_V = plt.figure(20)
    fig_lum_eff_at_2p5_V.set_size_inches(12,8)
    fig_lum_eff_at_2p5_V.set_tight_layout(True)
    ax_lum_eff_at_2p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 2.5 V [lm/W]'], ax=ax_lum_eff_at_2p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 2.5 V [lm/W]'], ax=ax_lum_eff_at_2p5_V, color='k', size=4)
    ax_lum_eff_at_2p5_V.set_xlabel(parameter_name)
    ax_lum_eff_at_2p5_V.set_ylabel('Luminous efficacy at 2.5 V [lm/W]')
    ax_lum_eff_at_2p5_V.set_title(experiment_name)
    ax_lum_eff_at_2p5_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth 
    
    ax_voltage_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_current_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_radiance_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_luminance_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_luminance_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_eqe_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_curr_eff_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_curr_eff_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_WP_eff_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_WP_eff_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_lum_eff_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_lum_eff_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    """Exporting plots"""
    plot_name = 'V at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_2p5_V.savefig(pdf_path)
    fig_voltage_at_2p5_V.savefig(svg_path)
    plt.close(fig_voltage_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'J at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_2p5_V.savefig(pdf_path)
    fig_current_at_2p5_V.savefig(svg_path)
    plt.close(fig_current_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'Rad at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_2p5_V.savefig(pdf_path)
    fig_radiance_at_2p5_V.savefig(svg_path)
    plt.close(fig_radiance_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_luminance_at_2p5_V.savefig(pdf_path)
    fig_luminance_at_2p5_V.savefig(svg_path)
    plt.close(fig_luminance_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'EQE at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_2p5_V.savefig(pdf_path)
    fig_eqe_at_2p5_V.savefig(svg_path)
    plt.close(fig_eqe_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path)) 
    
    plot_name = 'Curr eff at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_curr_eff_at_2p5_V.savefig(pdf_path)
    fig_curr_eff_at_2p5_V.savefig(svg_path)
    plt.close(fig_curr_eff_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'WP eff at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_WP_eff_at_2p5_V.savefig(pdf_path)
    fig_WP_eff_at_2p5_V.savefig(svg_path)
    plt.close(fig_WP_eff_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum eff at 2p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_lum_eff_at_2p5_V.savefig(pdf_path)
    fig_lum_eff_at_2p5_V.savefig(svg_path)
    plt.close(fig_lum_eff_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    """Voltage at 3.0 V"""
    fig_voltage_at_3p0_V = plt.figure(21)
    fig_voltage_at_3p0_V.set_size_inches(12,8)
    fig_voltage_at_3p0_V.set_tight_layout(True)
    ax_voltage_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.0 V [V]'], ax=ax_voltage_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.0 V [V]'], ax=ax_voltage_at_3p0_V, color='k', size=4)
    ax_voltage_at_3p0_V.set_xlabel(parameter_name)
    ax_voltage_at_3p0_V.set_ylabel('Voltage at 3.0 V [V]')
    ax_voltage_at_3p0_V.set_title(experiment_name)
    
    """Current Density at 3.0 V"""
    fig_current_at_3p0_V = plt.figure(22)
    fig_current_at_3p0_V.set_size_inches(12,8)
    fig_current_at_3p0_V.set_tight_layout(True)
    ax_current_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.0 V [mA/cm2]'], ax=ax_current_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.0 V [mA/cm2]'], ax=ax_current_at_3p0_V, color='k', size=4)
    ax_current_at_3p0_V.set_xlabel(parameter_name)
    ax_current_at_3p0_V.set_ylabel('Current Density at 3.0 V [mA/cm2]')
    ax_current_at_3p0_V.set_title(experiment_name)
    ax_current_at_3p0_V.set_ylim(bottom=0)
    
    """Radiance at 3.0 V"""
    fig_radiance_at_3p0_V = plt.figure(23)
    fig_radiance_at_3p0_V.set_size_inches(12,8)
    fig_radiance_at_3p0_V.set_tight_layout(True)
    ax_radiance_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.0 V [W/sr/m2]'], ax=ax_radiance_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.0 V [W/sr/m2]'], ax=ax_radiance_at_3p0_V, color='k', size=4)
    ax_radiance_at_3p0_V.set_xlabel(parameter_name)
    ax_radiance_at_3p0_V.set_ylabel('Radiance at 3.0 V [W/sr/m2]')
    ax_radiance_at_3p0_V.set_title(experiment_name)
    ax_radiance_at_3p0_V.set_ylim(bottom=0)
    
    """Luminance at 3.0 V"""
    fig_luminance_at_3p0_V = plt.figure(24)
    fig_luminance_at_3p0_V.set_size_inches(12,8)
    fig_luminance_at_3p0_V.set_tight_layout(True)
    ax_luminance_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.0 V [cd/m2]'], ax=ax_luminance_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.0 V [cd/m2]'], ax=ax_luminance_at_3p0_V, color='k', size=4)
    ax_luminance_at_3p0_V.set_xlabel(parameter_name)
    ax_luminance_at_3p0_V.set_ylabel('Luminance at 3.0 V [cd/m2]')
    ax_luminance_at_3p0_V.set_title(experiment_name)
    ax_luminance_at_3p0_V.set_ylim(bottom=0)
    
    """EQE at 3.0 V"""
    fig_eqe_at_3p0_V = plt.figure(25)
    fig_eqe_at_3p0_V.set_size_inches(12,8)
    fig_eqe_at_3p0_V.set_tight_layout(True)
    ax_eqe_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.0 V [%]'], ax=ax_eqe_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.0 V [%]'], ax=ax_eqe_at_3p0_V, color='k', size=4)
    ax_eqe_at_3p0_V.set_xlabel(parameter_name)
    ax_eqe_at_3p0_V.set_ylabel('EQE at 3.0 V [%]')
    ax_eqe_at_3p0_V.set_title(experiment_name)
    ax_eqe_at_3p0_V.set_ylim(bottom=0)
    
    """Current efficiency at 3.0 V"""
    fig_curr_eff_at_3p0_V = plt.figure(26)
    fig_curr_eff_at_3p0_V.set_size_inches(12,8)
    fig_curr_eff_at_3p0_V.set_tight_layout(True)
    ax_curr_eff_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.0 V [cd/A]'], ax=ax_curr_eff_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.0 V [cd/A]'], ax=ax_curr_eff_at_3p0_V, color='k', size=4)
    ax_curr_eff_at_3p0_V.set_xlabel(parameter_name)
    ax_curr_eff_at_3p0_V.set_ylabel('Current efficiency at 3.0 V [cd/A]')
    ax_curr_eff_at_3p0_V.set_title(experiment_name)
    ax_curr_eff_at_3p0_V.set_ylim(bottom=0)
    
    """Wall-plug efficiency at 3.0 V"""
    fig_WP_eff_at_3p0_V = plt.figure(27)
    fig_WP_eff_at_3p0_V.set_size_inches(12,8)
    fig_WP_eff_at_3p0_V.set_tight_layout(True)
    ax_WP_eff_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.0 V [W/W]'], ax=ax_WP_eff_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.0 V [W/W]'], ax=ax_WP_eff_at_3p0_V, color='k', size=4)
    ax_WP_eff_at_3p0_V.set_xlabel(parameter_name)
    ax_WP_eff_at_3p0_V.set_ylabel('Wall-plug efficiency at 3.0 V [W/W]')
    ax_WP_eff_at_3p0_V.set_title(experiment_name)
    ax_WP_eff_at_3p0_V.set_ylim(bottom=0)
    
    """Luminous efficacy at 3.0 V"""
    fig_lum_eff_at_3p0_V = plt.figure(28)
    fig_lum_eff_at_3p0_V.set_size_inches(12,8)
    fig_lum_eff_at_3p0_V.set_tight_layout(True)
    ax_lum_eff_at_3p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.0 V [lm/W]'], ax=ax_lum_eff_at_3p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.0 V [lm/W]'], ax=ax_lum_eff_at_3p0_V, color='k', size=4)
    ax_lum_eff_at_3p0_V.set_xlabel(parameter_name)
    ax_lum_eff_at_3p0_V.set_ylabel('Luminous efficacy at 3.0 V [lm/W]')
    ax_lum_eff_at_3p0_V.set_title(experiment_name)
    ax_lum_eff_at_3p0_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth 
    
    ax_voltage_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_current_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_radiance_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_luminance_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_luminance_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_eqe_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_curr_eff_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_curr_eff_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_WP_eff_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_WP_eff_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_lum_eff_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_lum_eff_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    """Exporting plots"""
    plot_name = 'V at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_3p0_V.savefig(pdf_path)
    fig_voltage_at_3p0_V.savefig(svg_path)
    plt.close(fig_voltage_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'J at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_3p0_V.savefig(pdf_path)
    fig_current_at_3p0_V.savefig(svg_path)
    plt.close(fig_current_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'Rad at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_3p0_V.savefig(pdf_path)
    fig_radiance_at_3p0_V.savefig(svg_path)
    plt.close(fig_radiance_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_luminance_at_3p0_V.savefig(pdf_path)
    fig_luminance_at_3p0_V.savefig(svg_path)
    plt.close(fig_luminance_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'EQE at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_3p0_V.savefig(pdf_path)
    fig_eqe_at_3p0_V.savefig(svg_path)
    plt.close(fig_eqe_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path)) 
    
    plot_name = 'Curr eff at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_curr_eff_at_3p0_V.savefig(pdf_path)
    fig_curr_eff_at_3p0_V.savefig(svg_path)
    plt.close(fig_curr_eff_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'WP eff at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_WP_eff_at_3p0_V.savefig(pdf_path)
    fig_WP_eff_at_3p0_V.savefig(svg_path)
    plt.close(fig_WP_eff_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum eff at 3p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_lum_eff_at_3p0_V.savefig(pdf_path)
    fig_lum_eff_at_3p0_V.savefig(svg_path)
    plt.close(fig_lum_eff_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    """Voltage at 3.25 V"""
    fig_voltage_at_3p25_V = plt.figure(29)
    fig_voltage_at_3p25_V.set_size_inches(12,8)
    fig_voltage_at_3p25_V.set_tight_layout(True)
    ax_voltage_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.25 V [V]'], ax=ax_voltage_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.25 V [V]'], ax=ax_voltage_at_3p25_V, color='k', size=4)
    ax_voltage_at_3p25_V.set_xlabel(parameter_name)
    ax_voltage_at_3p25_V.set_ylabel('Voltage at 3.25 V [V]')
    ax_voltage_at_3p25_V.set_title(experiment_name)
    
    """Current Density at 3.25 V"""
    fig_current_at_3p25_V = plt.figure(30)
    fig_current_at_3p25_V.set_size_inches(12,8)
    fig_current_at_3p25_V.set_tight_layout(True)
    ax_current_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.25 V [mA/cm2]'], ax=ax_current_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.25 V [mA/cm2]'], ax=ax_current_at_3p25_V, color='k', size=4)
    ax_current_at_3p25_V.set_xlabel(parameter_name)
    ax_current_at_3p25_V.set_ylabel('Current Density at 3.25 V [mA/cm2]')
    ax_current_at_3p25_V.set_title(experiment_name)
    ax_current_at_3p25_V.set_ylim(bottom=0)
    
    """Radiance at 3.25 V"""
    fig_radiance_at_3p25_V = plt.figure(31)
    fig_radiance_at_3p25_V.set_size_inches(12,8)
    fig_radiance_at_3p25_V.set_tight_layout(True)
    ax_radiance_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.25 V [W/sr/m2]'], ax=ax_radiance_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.25 V [W/sr/m2]'], ax=ax_radiance_at_3p25_V, color='k', size=4)
    ax_radiance_at_3p25_V.set_xlabel(parameter_name)
    ax_radiance_at_3p25_V.set_ylabel('Radiance at 3.25 V [W/sr/m2]')
    ax_radiance_at_3p25_V.set_title(experiment_name)
    ax_radiance_at_3p25_V.set_ylim(bottom=0)
    
    """Luminance at 3.25 V"""
    fig_luminance_at_3p25_V = plt.figure(32)
    fig_luminance_at_3p25_V.set_size_inches(12,8)
    fig_luminance_at_3p25_V.set_tight_layout(True)
    ax_luminance_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.25 V [cd/m2]'], ax=ax_luminance_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.25 V [cd/m2]'], ax=ax_luminance_at_3p25_V, color='k', size=4)
    ax_luminance_at_3p25_V.set_xlabel(parameter_name)
    ax_luminance_at_3p25_V.set_ylabel('Luminance at 3.25 V [cd/m2]')
    ax_luminance_at_3p25_V.set_title(experiment_name)
    ax_luminance_at_3p25_V.set_ylim(bottom=0)
    
    """EQE at 3.25 V"""
    fig_eqe_at_3p25_V = plt.figure(33)
    fig_eqe_at_3p25_V.set_size_inches(12,8)
    fig_eqe_at_3p25_V.set_tight_layout(True)
    ax_eqe_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.25 V [%]'], ax=ax_eqe_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.25 V [%]'], ax=ax_eqe_at_3p25_V, color='k', size=4)
    ax_eqe_at_3p25_V.set_xlabel(parameter_name)
    ax_eqe_at_3p25_V.set_ylabel('EQE at 3.25 V [%]')
    ax_eqe_at_3p25_V.set_title(experiment_name)
    ax_eqe_at_3p25_V.set_ylim(bottom=0)
    
    """Current efficiency at 3.25 V"""
    fig_curr_eff_at_3p25_V = plt.figure(34)
    fig_curr_eff_at_3p25_V.set_size_inches(12,8)
    fig_curr_eff_at_3p25_V.set_tight_layout(True)
    ax_curr_eff_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.25 V [cd/A]'], ax=ax_curr_eff_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.25 V [cd/A]'], ax=ax_curr_eff_at_3p25_V, color='k', size=4)
    ax_curr_eff_at_3p25_V.set_xlabel(parameter_name)
    ax_curr_eff_at_3p25_V.set_ylabel('Current efficiency at 3.25 V [cd/A]')
    ax_curr_eff_at_3p25_V.set_title(experiment_name)
    ax_curr_eff_at_3p25_V.set_ylim(bottom=0)
    
    """Wall-plug efficiency at 3.25 V"""
    fig_WP_eff_at_3p25_V = plt.figure(35)
    fig_WP_eff_at_3p25_V.set_size_inches(12,8)
    fig_WP_eff_at_3p25_V.set_tight_layout(True)
    ax_WP_eff_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.25 V [W/W]'], ax=ax_WP_eff_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.25 V [W/W]'], ax=ax_WP_eff_at_3p25_V, color='k', size=4)
    ax_WP_eff_at_3p25_V.set_xlabel(parameter_name)
    ax_WP_eff_at_3p25_V.set_ylabel('Wall-plug efficiency at 3.25 V [W/W]')
    ax_WP_eff_at_3p25_V.set_title(experiment_name)
    ax_WP_eff_at_3p25_V.set_ylim(bottom=0)
    
    """Luminous efficacy at 3.25 V"""
    fig_lum_eff_at_3p25_V = plt.figure(36)
    fig_lum_eff_at_3p25_V.set_size_inches(12,8)
    fig_lum_eff_at_3p25_V.set_tight_layout(True)
    ax_lum_eff_at_3p25_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.25 V [lm/W]'], ax=ax_lum_eff_at_3p25_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.25 V [lm/W]'], ax=ax_lum_eff_at_3p25_V, color='k', size=4)
    ax_lum_eff_at_3p25_V.set_xlabel(parameter_name)
    ax_lum_eff_at_3p25_V.set_ylabel('Luminous efficacy at 3.25 V [lm/W]')
    ax_lum_eff_at_3p25_V.set_title(experiment_name)
    ax_lum_eff_at_3p25_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth 
    
    ax_voltage_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_current_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_radiance_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_luminance_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_luminance_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_eqe_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_curr_eff_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_curr_eff_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_WP_eff_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_WP_eff_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_lum_eff_at_3p25_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p25_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_lum_eff_at_3p25_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p25_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    """Exporting plots"""
    plot_name = 'V at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_3p25_V.savefig(pdf_path)
    fig_voltage_at_3p25_V.savefig(svg_path)
    plt.close(fig_voltage_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'J at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_3p25_V.savefig(pdf_path)
    fig_current_at_3p25_V.savefig(svg_path)
    plt.close(fig_current_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'Rad at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_3p25_V.savefig(pdf_path)
    fig_radiance_at_3p25_V.savefig(svg_path)
    plt.close(fig_radiance_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_luminance_at_3p25_V.savefig(pdf_path)
    fig_luminance_at_3p25_V.savefig(svg_path)
    plt.close(fig_luminance_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'EQE at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_3p25_V.savefig(pdf_path)
    fig_eqe_at_3p25_V.savefig(svg_path)
    plt.close(fig_eqe_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path)) 
    
    plot_name = 'Curr eff at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_curr_eff_at_3p25_V.savefig(pdf_path)
    fig_curr_eff_at_3p25_V.savefig(svg_path)
    plt.close(fig_curr_eff_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'WP eff at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_WP_eff_at_3p25_V.savefig(pdf_path)
    fig_WP_eff_at_3p25_V.savefig(svg_path)
    plt.close(fig_WP_eff_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum eff at 3p25 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_lum_eff_at_3p25_V.savefig(pdf_path)
    fig_lum_eff_at_3p25_V.savefig(svg_path)
    plt.close(fig_lum_eff_at_3p25_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    """Voltage at 3.5 V"""
    fig_voltage_at_3p5_V = plt.figure(37)
    fig_voltage_at_3p5_V.set_size_inches(12,8)
    fig_voltage_at_3p5_V.set_tight_layout(True)
    ax_voltage_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.5 V [V]'], ax=ax_voltage_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.5 V [V]'], ax=ax_voltage_at_3p5_V, color='k', size=4)
    ax_voltage_at_3p5_V.set_xlabel(parameter_name)
    ax_voltage_at_3p5_V.set_ylabel('Voltage at 3.5 V [V]')
    ax_voltage_at_3p5_V.set_title(experiment_name)
    
    """Current Density at 3.5 V"""
    fig_current_at_3p5_V = plt.figure(38)
    fig_current_at_3p5_V.set_size_inches(12,8)
    fig_current_at_3p5_V.set_tight_layout(True)
    ax_current_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.5 V [mA/cm2]'], ax=ax_current_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.5 V [mA/cm2]'], ax=ax_current_at_3p5_V, color='k', size=4)
    ax_current_at_3p5_V.set_xlabel(parameter_name)
    ax_current_at_3p5_V.set_ylabel('Current Density at 3.5 V [mA/cm2]')
    ax_current_at_3p5_V.set_title(experiment_name)
    ax_current_at_3p5_V.set_ylim(bottom=0)
    
    """Radiance at 3.5 V"""
    fig_radiance_at_3p5_V = plt.figure(39)
    fig_radiance_at_3p5_V.set_size_inches(12,8)
    fig_radiance_at_3p5_V.set_tight_layout(True)
    ax_radiance_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.5 V [W/sr/m2]'], ax=ax_radiance_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.5 V [W/sr/m2]'], ax=ax_radiance_at_3p5_V, color='k', size=4)
    ax_radiance_at_3p5_V.set_xlabel(parameter_name)
    ax_radiance_at_3p5_V.set_ylabel('Radiance at 3.5 V [W/sr/m2]')
    ax_radiance_at_3p5_V.set_title(experiment_name)
    ax_radiance_at_3p5_V.set_ylim(bottom=0)
    
    """Luminance at 3.5 V"""
    fig_luminance_at_3p5_V = plt.figure(40)
    fig_luminance_at_3p5_V.set_size_inches(12,8)
    fig_luminance_at_3p5_V.set_tight_layout(True)
    ax_luminance_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.5 V [cd/m2]'], ax=ax_luminance_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.5 V [cd/m2]'], ax=ax_luminance_at_3p5_V, color='k', size=4)
    ax_luminance_at_3p5_V.set_xlabel(parameter_name)
    ax_luminance_at_3p5_V.set_ylabel('Luminance at 3.5 V [cd/m2]')
    ax_luminance_at_3p5_V.set_title(experiment_name)
    ax_luminance_at_3p5_V.set_ylim(bottom=0)
    
    """EQE at 3.5 V"""
    fig_eqe_at_3p5_V = plt.figure(41)
    fig_eqe_at_3p5_V.set_size_inches(12,8)
    fig_eqe_at_3p5_V.set_tight_layout(True)
    ax_eqe_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.5 V [%]'], ax=ax_eqe_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.5 V [%]'], ax=ax_eqe_at_3p5_V, color='k', size=4)
    ax_eqe_at_3p5_V.set_xlabel(parameter_name)
    ax_eqe_at_3p5_V.set_ylabel('EQE at 3.5 V [%]')
    ax_eqe_at_3p5_V.set_title(experiment_name)
    ax_eqe_at_3p5_V.set_ylim(bottom=0)
    
    """Current efficiency at 3.5 V"""
    fig_curr_eff_at_3p5_V = plt.figure(42)
    fig_curr_eff_at_3p5_V.set_size_inches(12,8)
    fig_curr_eff_at_3p5_V.set_tight_layout(True)
    ax_curr_eff_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.5 V [cd/A]'], ax=ax_curr_eff_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.5 V [cd/A]'], ax=ax_curr_eff_at_3p5_V, color='k', size=4)
    ax_curr_eff_at_3p5_V.set_xlabel(parameter_name)
    ax_curr_eff_at_3p5_V.set_ylabel('Current efficiency at 3.5 V [cd/A]')
    ax_curr_eff_at_3p5_V.set_title(experiment_name)
    ax_curr_eff_at_3p5_V.set_ylim(bottom=0)
    
    """Wall-plug efficiency at 3.5 V"""
    fig_WP_eff_at_3p5_V = plt.figure(43)
    fig_WP_eff_at_3p5_V.set_size_inches(12,8)
    fig_WP_eff_at_3p5_V.set_tight_layout(True)
    ax_WP_eff_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.5 V [W/W]'], ax=ax_WP_eff_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.5 V [W/W]'], ax=ax_WP_eff_at_3p5_V, color='k', size=4)
    ax_WP_eff_at_3p5_V.set_xlabel(parameter_name)
    ax_WP_eff_at_3p5_V.set_ylabel('Wall-plug efficiency at 3.5 V [W/W]')
    ax_WP_eff_at_3p5_V.set_title(experiment_name)
    ax_WP_eff_at_3p5_V.set_ylim(bottom=0)
    
    """Luminous efficacy at 3.5 V"""
    fig_lum_eff_at_3p5_V = plt.figure(44)
    fig_lum_eff_at_3p5_V.set_size_inches(12,8)
    fig_lum_eff_at_3p5_V.set_tight_layout(True)
    ax_lum_eff_at_3p5_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.5 V [lm/W]'], ax=ax_lum_eff_at_3p5_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.5 V [lm/W]'], ax=ax_lum_eff_at_3p5_V, color='k', size=4)
    ax_lum_eff_at_3p5_V.set_xlabel(parameter_name)
    ax_lum_eff_at_3p5_V.set_ylabel('Luminous efficacy at 3.5 V [lm/W]')
    ax_lum_eff_at_3p5_V.set_title(experiment_name)
    ax_lum_eff_at_3p5_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth 
    
    ax_voltage_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_current_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_radiance_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_luminance_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_luminance_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_eqe_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_curr_eff_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_curr_eff_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_WP_eff_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_WP_eff_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_lum_eff_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_lum_eff_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    """Exporting plots"""
    plot_name = 'V at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_3p5_V.savefig(pdf_path)
    fig_voltage_at_3p5_V.savefig(svg_path)
    plt.close(fig_voltage_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'J at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_3p5_V.savefig(pdf_path)
    fig_current_at_3p5_V.savefig(svg_path)
    plt.close(fig_current_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'Rad at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_3p5_V.savefig(pdf_path)
    fig_radiance_at_3p5_V.savefig(svg_path)
    plt.close(fig_radiance_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_luminance_at_3p5_V.savefig(pdf_path)
    fig_luminance_at_3p5_V.savefig(svg_path)
    plt.close(fig_luminance_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'EQE at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_3p5_V.savefig(pdf_path)
    fig_eqe_at_3p5_V.savefig(svg_path)
    plt.close(fig_eqe_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path)) 
    
    plot_name = 'Curr eff at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_curr_eff_at_3p5_V.savefig(pdf_path)
    fig_curr_eff_at_3p5_V.savefig(svg_path)
    plt.close(fig_curr_eff_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'WP eff at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_WP_eff_at_3p5_V.savefig(pdf_path)
    fig_WP_eff_at_3p5_V.savefig(svg_path)
    plt.close(fig_WP_eff_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum eff at 3p5 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_lum_eff_at_3p5_V.savefig(pdf_path)
    fig_lum_eff_at_3p5_V.savefig(svg_path)
    plt.close(fig_lum_eff_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    """Voltage at 3.75 V"""
    fig_voltage_at_3p75_V = plt.figure(45)
    fig_voltage_at_3p75_V.set_size_inches(12,8)
    fig_voltage_at_3p75_V.set_tight_layout(True)
    ax_voltage_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.75 V [V]'], ax=ax_voltage_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 3.75 V [V]'], ax=ax_voltage_at_3p75_V, color='k', size=4)
    ax_voltage_at_3p75_V.set_xlabel(parameter_name)
    ax_voltage_at_3p75_V.set_ylabel('Voltage at 3.75 V [V]')
    ax_voltage_at_3p75_V.set_title(experiment_name)
    
    """Current Density at 3.75 V"""
    fig_current_at_3p75_V = plt.figure(46)
    fig_current_at_3p75_V.set_size_inches(12,8)
    fig_current_at_3p75_V.set_tight_layout(True)
    ax_current_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.75 V [mA/cm2]'], ax=ax_current_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 3.75 V [mA/cm2]'], ax=ax_current_at_3p75_V, color='k', size=4)
    ax_current_at_3p75_V.set_xlabel(parameter_name)
    ax_current_at_3p75_V.set_ylabel('Current Density at 3.75 V [mA/cm2]')
    ax_current_at_3p75_V.set_title(experiment_name)
    ax_current_at_3p75_V.set_ylim(bottom=0)
    
    """Radiance at 3.75 V"""
    fig_radiance_at_3p75_V = plt.figure(47)
    fig_radiance_at_3p75_V.set_size_inches(12,8)
    fig_radiance_at_3p75_V.set_tight_layout(True)
    ax_radiance_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.75 V [W/sr/m2]'], ax=ax_radiance_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 3.75 V [W/sr/m2]'], ax=ax_radiance_at_3p75_V, color='k', size=4)
    ax_radiance_at_3p75_V.set_xlabel(parameter_name)
    ax_radiance_at_3p75_V.set_ylabel('Radiance at 3.75 V [W/sr/m2]')
    ax_radiance_at_3p75_V.set_title(experiment_name)
    ax_radiance_at_3p75_V.set_ylim(bottom=0)
    
    """Luminance at 3.75 V"""
    fig_luminance_at_3p75_V = plt.figure(48)
    fig_luminance_at_3p75_V.set_size_inches(12,8)
    fig_luminance_at_3p75_V.set_tight_layout(True)
    ax_luminance_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.75 V [cd/m2]'], ax=ax_luminance_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 3.75 V [cd/m2]'], ax=ax_luminance_at_3p75_V, color='k', size=4)
    ax_luminance_at_3p75_V.set_xlabel(parameter_name)
    ax_luminance_at_3p75_V.set_ylabel('Luminance at 3.75 V [cd/m2]')
    ax_luminance_at_3p75_V.set_title(experiment_name)
    ax_luminance_at_3p75_V.set_ylim(bottom=0)
    
    """EQE at 3.75 V"""
    fig_eqe_at_3p75_V = plt.figure(49)
    fig_eqe_at_3p75_V.set_size_inches(12,8)
    fig_eqe_at_3p75_V.set_tight_layout(True)
    ax_eqe_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.75 V [%]'], ax=ax_eqe_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 3.75 V [%]'], ax=ax_eqe_at_3p75_V, color='k', size=4)
    ax_eqe_at_3p75_V.set_xlabel(parameter_name)
    ax_eqe_at_3p75_V.set_ylabel('EQE at 3.75 V [%]')
    ax_eqe_at_3p75_V.set_title(experiment_name)
    ax_eqe_at_3p75_V.set_ylim(bottom=0)
    
    """Current efficiency at 3.75 V"""
    fig_curr_eff_at_3p75_V = plt.figure(50)
    fig_curr_eff_at_3p75_V.set_size_inches(12,8)
    fig_curr_eff_at_3p75_V.set_tight_layout(True)
    ax_curr_eff_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.75 V [cd/A]'], ax=ax_curr_eff_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 3.75 V [cd/A]'], ax=ax_curr_eff_at_3p75_V, color='k', size=4)
    ax_curr_eff_at_3p75_V.set_xlabel(parameter_name)
    ax_curr_eff_at_3p75_V.set_ylabel('Current efficiency at 3.75 V [cd/A]')
    ax_curr_eff_at_3p75_V.set_title(experiment_name)
    ax_curr_eff_at_3p75_V.set_ylim(bottom=0)
    
    """Wall-plug efficiency at 3.75 V"""
    fig_WP_eff_at_3p75_V = plt.figure(51)
    fig_WP_eff_at_3p75_V.set_size_inches(12,8)
    fig_WP_eff_at_3p75_V.set_tight_layout(True)
    ax_WP_eff_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.75 V [W/W]'], ax=ax_WP_eff_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 3.75 V [W/W]'], ax=ax_WP_eff_at_3p75_V, color='k', size=4)
    ax_WP_eff_at_3p75_V.set_xlabel(parameter_name)
    ax_WP_eff_at_3p75_V.set_ylabel('Wall-plug efficiency at 3.75 V [W/W]')
    ax_WP_eff_at_3p75_V.set_title(experiment_name)
    ax_WP_eff_at_3p75_V.set_ylim(bottom=0)
    
    """Luminous efficacy at 3.75 V"""
    fig_lum_eff_at_3p75_V = plt.figure(52)
    fig_lum_eff_at_3p75_V.set_size_inches(12,8)
    fig_lum_eff_at_3p75_V.set_tight_layout(True)
    ax_lum_eff_at_3p75_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.75 V [lm/W]'], ax=ax_lum_eff_at_3p75_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 3.75 V [lm/W]'], ax=ax_lum_eff_at_3p75_V, color='k', size=4)
    ax_lum_eff_at_3p75_V.set_xlabel(parameter_name)
    ax_lum_eff_at_3p75_V.set_ylabel('Luminous efficacy at 3.75 V [lm/W]')
    ax_lum_eff_at_3p75_V.set_title(experiment_name)
    ax_lum_eff_at_3p75_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth 
    
    ax_voltage_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_current_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_radiance_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_luminance_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_luminance_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_eqe_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_curr_eff_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_curr_eff_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_WP_eff_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_WP_eff_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_lum_eff_at_3p75_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p75_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_lum_eff_at_3p75_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_3p75_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    """Exporting plots"""
    plot_name = 'V at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_3p75_V.savefig(pdf_path)
    fig_voltage_at_3p75_V.savefig(svg_path)
    plt.close(fig_voltage_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'J at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_3p75_V.savefig(pdf_path)
    fig_current_at_3p75_V.savefig(svg_path)
    plt.close(fig_current_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'Rad at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_3p75_V.savefig(pdf_path)
    fig_radiance_at_3p75_V.savefig(svg_path)
    plt.close(fig_radiance_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_luminance_at_3p75_V.savefig(pdf_path)
    fig_luminance_at_3p75_V.savefig(svg_path)
    plt.close(fig_luminance_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'EQE at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_3p75_V.savefig(pdf_path)
    fig_eqe_at_3p75_V.savefig(svg_path)
    plt.close(fig_eqe_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path)) 
    
    plot_name = 'Curr eff at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_curr_eff_at_3p75_V.savefig(pdf_path)
    fig_curr_eff_at_3p75_V.savefig(svg_path)
    plt.close(fig_curr_eff_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'WP eff at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_WP_eff_at_3p75_V.savefig(pdf_path)
    fig_WP_eff_at_3p75_V.savefig(svg_path)
    plt.close(fig_WP_eff_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum eff at 3p75 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_lum_eff_at_3p75_V.savefig(pdf_path)
    fig_lum_eff_at_3p75_V.savefig(svg_path)
    plt.close(fig_lum_eff_at_3p75_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    """Voltage at 4.0 V"""
    fig_voltage_at_4p0_V = plt.figure(53)
    fig_voltage_at_4p0_V.set_size_inches(12,8)
    fig_voltage_at_4p0_V.set_tight_layout(True)
    ax_voltage_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 4.0 V [V]'], ax=ax_voltage_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Voltage at 4.0 V [V]'], ax=ax_voltage_at_4p0_V, color='k', size=4)
    ax_voltage_at_4p0_V.set_xlabel(parameter_name)
    ax_voltage_at_4p0_V.set_ylabel('Voltage at 4.0 V [V]')
    ax_voltage_at_4p0_V.set_title(experiment_name)
    
    """Current Density at 4.0 V"""
    fig_current_at_4p0_V = plt.figure(54)
    fig_current_at_4p0_V.set_size_inches(12,8)
    fig_current_at_4p0_V.set_tight_layout(True)
    ax_current_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 4.0 V [mA/cm2]'], ax=ax_current_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current Density at 4.0 V [mA/cm2]'], ax=ax_current_at_4p0_V, color='k', size=4)
    ax_current_at_4p0_V.set_xlabel(parameter_name)
    ax_current_at_4p0_V.set_ylabel('Current Density at 4.0 V [mA/cm2]')
    ax_current_at_4p0_V.set_title(experiment_name)
    ax_current_at_4p0_V.set_ylim(bottom=0)
    
    """Radiance at 4.0 V"""
    fig_radiance_at_4p0_V = plt.figure(55)
    fig_radiance_at_4p0_V.set_size_inches(12,8)
    fig_radiance_at_4p0_V.set_tight_layout(True)
    ax_radiance_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 4.0 V [W/sr/m2]'], ax=ax_radiance_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Radiance at 4.0 V [W/sr/m2]'], ax=ax_radiance_at_4p0_V, color='k', size=4)
    ax_radiance_at_4p0_V.set_xlabel(parameter_name)
    ax_radiance_at_4p0_V.set_ylabel('Radiance at 4.0 V [W/sr/m2]')
    ax_radiance_at_4p0_V.set_title(experiment_name)
    ax_radiance_at_4p0_V.set_ylim(bottom=0)
    
    """Luminance at 4.0 V"""
    fig_luminance_at_4p0_V = plt.figure(56)
    fig_luminance_at_4p0_V.set_size_inches(12,8)
    fig_luminance_at_4p0_V.set_tight_layout(True)
    ax_luminance_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 4.0 V [cd/m2]'], ax=ax_luminance_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminance at 4.0 V [cd/m2]'], ax=ax_luminance_at_4p0_V, color='k', size=4)
    ax_luminance_at_4p0_V.set_xlabel(parameter_name)
    ax_luminance_at_4p0_V.set_ylabel('Luminance at 4.0 V [cd/m2]')
    ax_luminance_at_4p0_V.set_title(experiment_name)
    ax_luminance_at_4p0_V.set_ylim(bottom=0)
    
    """EQE at 4.0 V"""
    fig_eqe_at_4p0_V = plt.figure(57)
    fig_eqe_at_4p0_V.set_size_inches(12,8)
    fig_eqe_at_4p0_V.set_tight_layout(True)
    ax_eqe_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 4.0 V [%]'], ax=ax_eqe_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['EQE at 4.0 V [%]'], ax=ax_eqe_at_4p0_V, color='k', size=4)
    ax_eqe_at_4p0_V.set_xlabel(parameter_name)
    ax_eqe_at_4p0_V.set_ylabel('EQE at 4.0 V [%]')
    ax_eqe_at_4p0_V.set_title(experiment_name)
    ax_eqe_at_4p0_V.set_ylim(bottom=0)
    
    """Current efficiency at 4.0 V"""
    fig_curr_eff_at_4p0_V = plt.figure(58)
    fig_curr_eff_at_4p0_V.set_size_inches(12,8)
    fig_curr_eff_at_4p0_V.set_tight_layout(True)
    ax_curr_eff_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 4.0 V [cd/A]'], ax=ax_curr_eff_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Current efficiency at 4.0 V [cd/A]'], ax=ax_curr_eff_at_4p0_V, color='k', size=4)
    ax_curr_eff_at_4p0_V.set_xlabel(parameter_name)
    ax_curr_eff_at_4p0_V.set_ylabel('Current efficiency at 4.0 V [cd/A]')
    ax_curr_eff_at_4p0_V.set_title(experiment_name)
    ax_curr_eff_at_4p0_V.set_ylim(bottom=0)
    
    """Wall-plug efficiency at 4.0 V"""
    fig_WP_eff_at_4p0_V = plt.figure(59)
    fig_WP_eff_at_4p0_V.set_size_inches(12,8)
    fig_WP_eff_at_4p0_V.set_tight_layout(True)
    ax_WP_eff_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 4.0 V [W/W]'], ax=ax_WP_eff_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Wall-plug efficiency at 4.0 V [W/W]'], ax=ax_WP_eff_at_4p0_V, color='k', size=4)
    ax_WP_eff_at_4p0_V.set_xlabel(parameter_name)
    ax_WP_eff_at_4p0_V.set_ylabel('Wall-plug efficiency at 4.0 V [W/W]')
    ax_WP_eff_at_4p0_V.set_title(experiment_name)
    ax_WP_eff_at_4p0_V.set_ylim(bottom=0)
    
    """Luminous efficacy at 4.0 V"""
    fig_lum_eff_at_4p0_V = plt.figure(60)
    fig_lum_eff_at_4p0_V.set_size_inches(12,8)
    fig_lum_eff_at_4p0_V.set_tight_layout(True)
    ax_lum_eff_at_4p0_V = plt.gca()  
    sns.boxplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 4.0 V [lm/W]'], ax=ax_lum_eff_at_4p0_V, fliersize=0, **boxplot_kwargs)
    sns.stripplot(x=big_data_df[parameter_name], y=big_data_df['Luminous efficacy at 4.0 V [lm/W]'], ax=ax_lum_eff_at_4p0_V, color='k', size=4)
    ax_lum_eff_at_4p0_V.set_xlabel(parameter_name)
    ax_lum_eff_at_4p0_V.set_ylabel('Luminous efficacy at 4.0 V [lm/W]')
    ax_lum_eff_at_4p0_V.set_title(experiment_name)
    ax_lum_eff_at_4p0_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth 
    
    ax_voltage_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_current_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_radiance_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_luminance_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_luminance_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_luminance_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_eqe_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)  
    
    ax_curr_eff_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_curr_eff_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_curr_eff_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_WP_eff_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_WP_eff_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_WP_eff_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    ax_lum_eff_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_lum_eff_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_lum_eff_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)   
    
    """Exporting plots"""
    plot_name = 'V at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_voltage_at_4p0_V.savefig(pdf_path)
    fig_voltage_at_4p0_V.savefig(svg_path)
    plt.close(fig_voltage_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'J at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_current_at_4p0_V.savefig(pdf_path)
    fig_current_at_4p0_V.savefig(svg_path)
    plt.close(fig_current_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))  
    
    plot_name = 'Rad at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_radiance_at_4p0_V.savefig(pdf_path)
    fig_radiance_at_4p0_V.savefig(svg_path)
    plt.close(fig_radiance_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_luminance_at_4p0_V.savefig(pdf_path)
    fig_luminance_at_4p0_V.savefig(svg_path)
    plt.close(fig_luminance_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'EQE at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_eqe_at_4p0_V.savefig(pdf_path)
    fig_eqe_at_4p0_V.savefig(svg_path)
    plt.close(fig_eqe_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path)) 
    
    plot_name = 'Curr eff at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_curr_eff_at_4p0_V.savefig(pdf_path)
    fig_curr_eff_at_4p0_V.savefig(svg_path)
    plt.close(fig_curr_eff_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'WP eff at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_WP_eff_at_4p0_V.savefig(pdf_path)
    fig_WP_eff_at_4p0_V.savefig(svg_path)
    plt.close(fig_WP_eff_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = 'Lum eff at 4p0 V box-strip plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    if len(pdf_path) > 255:
        warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
        write_to_log(warning_string)
    fig_lum_eff_at_4p0_V.savefig(pdf_path)
    fig_lum_eff_at_4p0_V.savefig(svg_path)
    plt.close(fig_lum_eff_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    print('Exported big data .pdf and .svg plots for: %s' % experiment_name)  

"""Plotting summary plots for the timed analysis"""
def plot_timed_data(timed_summary_df,experiment_name): # modified relative to the original
    """Some constants for graphical settings"""
    scalar_ticksize_major = 5
    scalar_ticksize_minor = 0
    scalar_tickwidth_minor = 0
    scalar_linewidth = 2
    
    """Box swarm plot general graphic settings"""
    boxcolour = COLOURS_GREENS[700]
    boxprops = {'edgecolor': boxcolour, 'linewidth': scalar_linewidth, 'facecolor': 'w'}
    lineprops = {'color': boxcolour, 'linewidth': scalar_linewidth}
    boxplot_kwargs = dict({'boxprops': boxprops, 'medianprops': lineprops,
                           'whiskerprops': lineprops, 'capprops': lineprops,
                           'width': 0.75})
    
    """Plotting all the box swarm plots"""
    """Max EQE"""
    fig_max_eqe = plt.figure(4)
    fig_max_eqe.set_size_inches(9,8)
    fig_max_eqe.set_tight_layout(True)
    ax_max_eqe = plt.gca()
    sns.boxplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Max EQE [%]'], ax=ax_max_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Max EQE [%]'], ax=ax_max_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_max_eqe.set_xlabel(parameter_name)
    ax_max_eqe.set_ylabel('Max EQE [%]')
    ax_max_eqe.set_title(experiment_name)
    ax_max_eqe.set_ylim(bottom=0)
    if timed_summary_df['Max EQE [%]'].max() > 25:
        ax_max_eqe.set_ylim(top=25)
        
    """Time to reach max EQE"""
    fig_time_to_max = plt.figure(5)
    fig_time_to_max.set_size_inches(9,8)
    fig_time_to_max.set_tight_layout(True)
    ax_time_to_max = plt.gca()
    sns.boxplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to max EQE [s]'], ax=ax_time_to_max, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to max EQE [s]'], ax=ax_time_to_max, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_max.set_xlabel(parameter_name)
    ax_time_to_max.set_ylabel('Time to max EQE [s]')
    ax_time_to_max.set_title(experiment_name)
    ax_time_to_max.set_ylim(bottom=0)
        
    """Time to 50% of max EQE from time of max"""
    fig_time_to_50_max = plt.figure(6)
    fig_time_to_50_max.set_size_inches(9,8)
    fig_time_to_50_max.set_tight_layout(True)
    ax_time_to_50_max = plt.gca()
    sns.boxplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to 50% of max EQE from time of max [s]'], ax=ax_time_to_50_max, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df[parameter_name][timed_summary_df['Time to 50% of max EQE from time of max [s]'].notna()], y=timed_summary_df['Time to 50% of max EQE from time of max [s]'][timed_summary_df['Time to 50% of max EQE from time of max [s]'].notna()], ax=ax_time_to_50_max, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_50_max.set_xlabel(parameter_name)
    ax_time_to_50_max.set_ylabel('Time to 50% of max EQE from time of max [s]')
    ax_time_to_50_max.set_title(experiment_name)
    ax_time_to_50_max.set_ylim(bottom=0)    
    # This .notna() solution removes the NaNs before plotting, which somehow solves an issue with edgecolor=boxcolour in the swarmplot for these two plots only with the dummy data. The error was of type "ValueError: Invalid RGBA argument". I think it has something to do with an arraysize related to boxcolour (e.g. setting edgecolor='k' solves it), but since boxcolour is a constant for all the values and works well in all other plots (despite those having NaNs too, but less), I do not understand exactly what goes wrong.
    
    """Time to 50% of intitial EQE from start"""
    fig_time_to_50_initial = plt.figure(7)
    fig_time_to_50_initial.set_size_inches(9,8)
    fig_time_to_50_initial.set_tight_layout(True)
    ax_time_to_50_initial = plt.gca()
    sns.boxplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to 50% of initial EQE from start [s]'], ax=ax_time_to_50_initial, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df[parameter_name][timed_summary_df['Time to 50% of initial EQE from start [s]'].notna()], y=timed_summary_df['Time to 50% of initial EQE from start [s]'][timed_summary_df['Time to 50% of initial EQE from start [s]'].notna()], ax=ax_time_to_50_initial, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_50_initial.set_xlabel(parameter_name)
    ax_time_to_50_initial.set_ylabel('Time to 50% of initial EQE from start [s]')
    ax_time_to_50_initial.set_title(experiment_name)
    ax_time_to_50_initial.set_ylim(bottom=0)
    # This .notna() solution removes the NaNs before plotting, which somehow solves an issue with edgecolor=boxcolour in the swarmplot for these two plots only with the dummy data. The error was of type "ValueError: Invalid RGBA argument". I think it has something to do with an arraysize related to boxcolour (e.g. setting edgecolor='k' solves it), but since boxcolour is a constant for all the values and works well in all other plots (despite those having NaNs too, but less), I do not understand exactly what goes wrong.
    
    """Time to 80% of max EQE from time of max"""
    fig_time_to_80_max = plt.figure(8)
    fig_time_to_80_max.set_size_inches(9,8)
    fig_time_to_80_max.set_tight_layout(True)
    ax_time_to_80_max = plt.gca()
    sns.boxplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to 80% of max EQE from time of max [s]'], ax=ax_time_to_80_max, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to 80% of max EQE from time of max [s]'], ax=ax_time_to_80_max, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_80_max.set_xlabel(parameter_name)
    ax_time_to_80_max.set_ylabel('Time to 80% of max EQE from time of max [s]')
    ax_time_to_80_max.set_title(experiment_name)
    ax_time_to_80_max.set_ylim(bottom=0)
        
    """Time to 80% of intitial EQE from start"""
    fig_time_to_80_initial = plt.figure(9)
    fig_time_to_80_initial.set_size_inches(9,8)
    fig_time_to_80_initial.set_tight_layout(True)
    ax_time_to_80_initial = plt.gca()
    sns.boxplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to 80% of initial EQE from start [s]'], ax=ax_time_to_80_initial, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df[parameter_name], y=timed_summary_df['Time to 80% of initial EQE from start [s]'], ax=ax_time_to_80_initial, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_80_initial.set_xlabel(parameter_name)
    ax_time_to_80_initial.set_ylabel('Time to 80% of initial EQE from start [s]')
    ax_time_to_80_initial.set_title(experiment_name)
    ax_time_to_80_initial.set_ylim(bottom=0)
      
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth  
    
    ax_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_time_to_max.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_max.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_max.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_max.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_time_to_50_max.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_max.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_50_max.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_max.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        
    ax_time_to_50_initial.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_initial.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_50_initial.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_initial.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_time_to_80_max.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_max.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_80_max.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_max.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
           
    ax_time_to_80_initial.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_initial.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_80_initial.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_initial.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        
    """Exporting plots"""
    plot_name = 'Timed Max EQE box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_max_eqe.savefig(pdf_path)
    fig_max_eqe.savefig(svg_path)
    plt.close(fig_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = 'Timed time-to-max box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_max.savefig(pdf_path)
    fig_time_to_max.savefig(svg_path)
    plt.close(fig_time_to_max)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = 'Timed t50-from-max box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_50_max.savefig(pdf_path)
    fig_time_to_50_max.savefig(svg_path)
    plt.close(fig_time_to_50_max)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = 'Timed t50-initial box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_50_initial.savefig(pdf_path)
    fig_time_to_50_initial.savefig(svg_path)
    plt.close(fig_time_to_50_initial)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = 'Timed t80-from-max box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_80_max.savefig(pdf_path)
    fig_time_to_80_max.savefig(svg_path)
    plt.close(fig_time_to_80_max)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = 'Timed t80-initial box-swarm plot'
    export_name_pdf = plot_name + '.pdf'
    export_name_svg = plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_80_initial.savefig(pdf_path)
    fig_time_to_80_initial.savefig(svg_path)
    plt.close(fig_time_to_80_initial)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))        
        
    print('Exported timed summary .pdf and .svg plots for: %s' % experiment_name)    
    return

"""Compiler for analysing scan type data on a pixel level"""
def compiler_scan(): # modified relative to the original
    print('Plotting scan type measurements.') 
    write_to_log('Plotting scan type measurements.')
    
    scan_summary_path = os.path.join(main_folder_path,scan_summary_name)
    scan_summary_df = pd.read_csv(scan_summary_path)
    
    global radiance_threshold
    if (scan_summary_df['Luminescence Turn-on radiance threshold [W/sr/m2]'][0] == scan_summary_df['Luminescence Turn-on radiance threshold [W/sr/m2]']).all(): # true if only 1 unique value
        radiance_threshold = scan_summary_df['Luminescence Turn-on radiance threshold [W/sr/m2]'][0]
    else:
        print('Warning! More than one unique radiance threshold value found. Plotting only one of the values.') 
        write_to_log('Warning! More than one unique radiance threshold value found. Plotting only one of the values.')
        radiance_threshold = scan_summary_df['Luminescence Turn-on radiance threshold [W/sr/m2]'][0]
    
    """Plot values from the scan summary analysis"""
    plot_scan_summary(scan_summary_df,scan_summary_name)

"""Compiler for analysing scan type data on an individual scan level"""
def compiler_scan_big_data(): # modified relative to the original
    print('Plotting big data scan type measurements.') 
    write_to_log('Plotting big data scan type measurements.')
    
    big_data_summary_path = os.path.join(main_folder_path,big_data_summary_name)
    big_data_summary_df = pd.read_csv(big_data_summary_path)    
    
    """Plot values from the big data analyis"""
    plot_big_data(big_data_summary_df,big_data_summary_name)
    
"""Compiler for analysing timed type data"""
def compiler_timed(): # modified relative to the original
    print('Plotting timed type measurements.') 
    write_to_log('Plotting timed type measurements.')
    
    timed_summary_path = os.join(main_folder_path,timed_summary_name)
    timed_summary_df = pd.read_csv(timed_summary_path)
       
    """Plot values from the scan summary analysis"""
    plot_timed_data(timed_summary_df,timed_summary_name)
    
"""Compiling the functions of the program"""
def compiler_main(): # modified relative to the original
    """Collecting user-defined values for these constants and making them global"""   
    global measurement_type
    global scan_summary_name
    global big_data_summary_name
    global timed_summary_name
    global parameter_name
    global export_folder_name
    global time_prompt
    [measurement_type, scan_summary_name, big_data_summary_name, timed_summary_name, parameter_name, export_folder_name] = prompt_user_input()
    time_prompt = datetime.datetime.now()
   
    """"Setting booleans for the different measurement types (scan/timed) and making them global"""
    global type_scan
    global type_timed
    if measurement_type == 1:
        type_scan = True
    elif measurement_type == 2:
        type_scan = False
        
    type_timed = not type_scan
    
    """"Create folder structure and log file"""
    global main_folder_path
    global data_folder_path
    global export_folder_path
    main_folder_path = os.path.dirname(os.path.abspath(__file__)) # find folder of py file
    main_folder_path = '\\\\?\\' + main_folder_path # resolves an issue with paths having more than 260 characters
#    export_folder_path = os.path.join(main_folder_path,export_folder_name) # create subfolder for exporting # temporarily changing this to avoid too long filepaths for Adobe Acrobat to handle
    export_folder_path = main_folder_path # temporary solution
# =============================================================================
#     if not os.path.exists(export_folder_path):
#         os.mkdir(export_folder_path) # create folder if non-existent
# =============================================================================
    
    if type_scan:
        global subfolder_big_data_plots_path
        subfolder_big_data_plots_name = 'Big data plots (scans)'
        global subfolder_big_data_plots_path
        subfolder_big_data_plots_path = os.path.join(export_folder_path,subfolder_big_data_plots_name)
        if not os.path.exists(subfolder_big_data_plots_path):
            os.mkdir(subfolder_big_data_plots_path) # create folder if non-existent
        
    global subfolder_summary_plots_name
    if type_scan:
        subfolder_summary_plots_name = 'Summary plots (scans)'
    elif type_timed:
        subfolder_summary_plots_name = 'Summary plots (timed)'
   
    global subfolder_summary_plots_path
    subfolder_summary_plots_path = os.path.join(export_folder_path,subfolder_summary_plots_name)
    if not os.path.exists(subfolder_summary_plots_path):
        os.mkdir(subfolder_summary_plots_path) # create folder if non-existent
    
    print('\nCreated export folders.')  
     
    global log_file_path    
    log_file_name = export_folder_name + ' log.txt'
    log_file_path = os.path.join(export_folder_path,log_file_name)
    log_file = open(log_file_path,'w')
    log_file.close()
    print('Created log file.\n')
    
    """Export initial information to log"""
    strings_to_log_initial = ['Program started: ' + time_start.strftime('%Y-%m-%d, %H:%M:%S'),\
                              'Successful prompt: ' + time_prompt.strftime('%Y-%m-%d, %H:%M:%S') + ' (Elapsed time: ' + str((time_prompt-time_start).total_seconds()) + ' s)',\
                                  '----------',\
                                      'Information prompted:',\
                                          'Measurement type (scans = 1, timed = 2): ' + str(measurement_type),\
                                              'Name of scan summary csv file: ' + str(scan_summary_name),\
                                                  'Name of big data summary csv file: ' + str(big_data_summary_name),\
                                                      'Name of timed summary csv file: ' + str(timed_summary_name),\
                                                          'Parameter column name: ' + str(parameter_name),\
                                                              'Export folder name: ' + export_folder_name,\
                                                                              '----------',\
                                                                                  'Created export folder: ' + str(export_folder_path),\
                                                                                          '----------']
    write_to_log(strings_to_log_initial)
       
    """Analyse and export data"""
    if type_scan:
        if scan_summary_name != 0:
            compiler_scan()
            if big_data_summary_name != 0:
                write_to_log('----------')
        
        if big_data_summary_name != 0:
            compiler_scan_big_data()
    
    elif type_timed:
        compiler_timed()
    
    global time_end
    time_end = datetime.datetime.now()        
    strings_to_log_end = ['----------',\
                              'Job completed ' + time_end.strftime('%Y-%m-%d, %H:%M:%S') + ' (Elapsed time: ' + str((time_end-time_start).total_seconds()) + ' s)']
    write_to_log(strings_to_log_end)
    print('Job completed.')

compiler_main()